import os

import infra.basetest


class TestMake(infra.basetest.BRTest):
    rootfs_overlay = \
        infra.filepath("tests/package/test_make/rootfs-overlay")
    config = infra.basetest.BASIC_TOOLCHAIN_CONFIG + \
        f"""
        BR2_PACKAGE_MAKE=y
        BR2_ROOTFS_OVERLAY="{rootfs_overlay}"
        BR2_TARGET_ROOTFS_CPIO=y
        # BR2_TARGET_ROOTFS_TAR is not set
        """

    def gen_expected_str(self, count):
        """Return the expected string generated by the test Makefile"""
        return "".join(map(lambda x: str(x), range(1, count+1)))

    def test_run(self):
        cpio_file = os.path.join(self.builddir, "images", "rootfs.cpio")
        self.emulator.boot(arch="armv5",
                           kernel="builtin",
                           options=["-initrd", cpio_file])
        self.emulator.login()

        # Check the program can execute.
        self.assertRunOk("make --version")

        # We touch the Makefile to set its modification time to the
        # current system time. This is to avoid warnings from Make
        # about having files with timestamps in the future. This is
        # because the minimal system running in the emulator might not
        # set the clock to the real time, and the Makefile has a
        # correct timestamp from the build host (which is likely at
        # the correct time).
        self.assertRunOk("touch Makefile")

        # We test the "message" target and check we get the expected
        # string.
        out, ret = self.emulator.run("make message")
        self.assertEqual(ret, 0)
        self.assertEqual(out[0], "Hello Buildroot!")

        # We redo the same test, this time by passing a new message
        # with a variable.
        msg = "This is Another Message..."
        out, ret = self.emulator.run(f"make message MESSAGE='{msg}'")
        self.assertEqual(ret, 0)
        self.assertEqual(out[0], msg)

        # We run a simple "make" invocation, using the defaults.
        self.assertRunOk("make")

        # We check the generated output contains the expected string.
        expected_str = self.gen_expected_str(10)
        out, ret = self.emulator.run("cat output.txt")
        self.assertEqual(ret, 0)
        self.assertEqual(out[0], expected_str)

        # Clean the previous invocation.
        self.assertRunOk("make clean")

        # We check a output generated file is no longer present.
        self.assertRunOk("test ! -e output.txt")

        # We run an invocation with a larger COUNT value. GNU Make
        # version 4.4 introduced the --shuffle option, which shuffle
        # rules. We use it with a constant seed, in order to have a
        # stable reshuffling in all test runs. We also include in this
        # execution a request for parallel jobs.
        count = 50
        seed = 123456
        self.assertRunOk(f"make -j10 --shuffle={seed} COUNT={count}")

        # Despite the pseudo-randomization in the previous invocation,
        # the expected output should be correctly ordered.
        expected_str = self.gen_expected_str(count)
        out, ret = self.emulator.run("cat output.txt")
        self.assertEqual(ret, 0)
        self.assertEqual(out[0], expected_str)
